#!/bin/env ruby
# encoding: utf-8
require 'rubygems'
require 'rspec'
require 'watir-webdriver'
require 'page-object'
require 'oci8'
require 'watir-scroll'

require_relative '../../module/DriverUtility'
require_relative '../rspec_helper'
require_relative './request_helper'
require_relative '../../pages/landing'
require_relative '../../pages/login'
require_relative '../../pages/eula'
require_relative '../../common/common'
require_relative '../../pages/landing_appointments'
require_relative '../../pages/appointments/new_appointment_request'
require_relative '../../pages/appointments/request_appointment_detail'
require_relative '../../helper/modal'
require_relative '../../module/database/OracleUtility'
require_relative '../../module/DateUtility'
require_relative '../../helper/error_validation'
require_relative '../../module/database/MongoUtility'

describe '[Story# VAR-1920: Appointment Request - Message Clerk]
          [Story# VAR-1921: New Appointment/Request – Appointment vs. Request Routing Question Updates (5.2.3.c)]' do
  include DriverUtility
  include OracleUtility
  include DateUtility
  include MongoUtility

  before(:all) do
    initializeConfigurations(VAR_BASE_URL)
    @landing_appointments = LandingAppointments.new(@driver)
    @login = Login.new(@driver)
    @eula = Eula.new(@driver)
    @common = Common.new(@driver)
    @modal = Modal.new(@driver)
    @new_request = NewAppointmentRequest.new(@driver)
    @detail = RequestAppointmentDetail.new(@driver)

    APPOINTMENTS = "Appointments/Requests"

    #mongo
    @db = connectToMongoDB("var-utility")
    restoreCollection("siteSupportingVAR", "var-utility", "sites-supporting-var-exported2.json", @db)
    restoreCollection("clinical-services", "var-utility", "clinical_services_default.json", @db)
    restoreCollection("requestEligibilityCriteria", "var-utility", "requestEligibilityCriteria.json", @db)
    @new_message_50 = "01234567890123456789012345678901234567890123456789"
    @today = getDateNthDaysFromNow(0, "%m/%d/%Y")
    @patientID = "1006088937V099668"
    @patientID2 = "1113138327"
    deletePatientData(@patientID)
    deletePatientData(@patientID2)
    @common.loginEndsOnHome(pw_redacted)
    @landing_appointments.primary_header_element.when_present(TIME_OUT_LIMIT)

  end

  after(:all) do
    deletePatientData(@patientID)
    deletePatientData(@patientID2)
    @driver.close
  end

  context 'AC#VAR-2045|TC#VAR-2221: Appointment Requests - Message Clerk ' do
    it "Verify request is created" do

      #add clerk Mental Health
      typeOfCare = "Outpatient Mental Health"
      schedulingMethod = "clerk"
      facilityId = "523"
      facilityName = "BOSTON HCS VAMC"
      typeOfVisit = "Office Visit"
      purposeOfVisit = "Medication Concern"
      otherPurposeOfVisit = ""
      desiredDateTime1 = [getDateNthDaysFromNow(40, "%m/%d/%Y"), "AM"]
      desiredDateTime2 = [getDateNthDaysFromNow(42, "%m/%d/%Y"), "PM"]
      desiredDAteTime3 = [getDateNthDaysFromNow(43, "%m/%d/%Y"), "AM"]
      phoneNum = "(234) 567-8901"
      verifyPhoneNum = "(234) 567-8901"

      bestTimeToCall = " Afternoon, Evening"
      call_morning = false
      call_afternoon = true
      call_evening = true
      bestTimesForVAtoCall = [call_morning, call_afternoon, call_evening]
      vaToCallYou = false
      message = ""
      sendMeEmail = "false"
      email = "test1@hotmail.com"

      addRequest(schedulingMethod, typeOfCare, facilityId, typeOfVisit, purposeOfVisit, otherPurposeOfVisit, desiredDateTime1, desiredDateTime2, desiredDAteTime3, phoneNum, verifyPhoneNum, bestTimesForVAtoCall, vaToCallYou, message, sendMeEmail, email  )
      verifyTableList(facilityName, @today, typeOfCare)

    end

    it "Verify message a scheduling clerk section is present and details" do

      @landing_appointments.request_table_list_elements[0].click
      @detail.status_element.when_present(TIME_OUT_LIMIT)
      @detail.message_desc_element.when_visible(TIME_OUT_LIMIT)
      expect(@detail.send_btn_element.disabled?).to eq(true)
      expect(@detail.message_header).to eq("Message a Scheduling Clerk")
      expect(@detail.message_desc).to eq("You may send up to two (2) messages while this request is in Submitted status. If you need additional information, please call your local VA Medical Center")
      expect(@detail.new_message_element.attribute("placeholder")).to eq("Messages (100 characters max) should not be used for cancelling an appointment or for clinical questions that may need provider attention.")
      expect(@detail.message_counter).to eq("100 characters remaining")

    end

    it "Verify character remaining texts when message entered with 50 chars" do
      @detail.new_message= @new_message_50
      expect(@detail.message_counter).to eq("50 characters remaining")
    end

    it "Verify character remaining texts when message entered with 100 chars" do
      @detail.new_message= @new_message_50 + @new_message_50
      expect(@detail.new_message).to eq(@new_message_50 + @new_message_50)
      expect(@detail.message_counter).to eq("0 characters remaining")
    end

    it "Verify more than 100 chars cannot be entered" do
      @detail.new_message= @new_message_50 + @new_message_50 + "!!@#!@#!@#%^^"
      expect(@detail.new_message).to eq(@new_message_50 + @new_message_50)
      expect(@detail.message_counter).to eq("0 characters remaining")
    end

    it "Send button is enabled when message entered" do
      @detail.send_btn_element.when_present(TIME_OUT_LIMIT)
      expect(@detail.send_btn_element.present?).to eq(true)
      expect(@detail.send_btn_element.disabled?).to eq(false)
    end

    it "Verify sent message text is displayed in details" do
      @detail.send_btn
      @common.waitWhileSpinnerPresent
      @detail.message_section_element.when_present(TIME_OUT_LIMIT)
      Watir::Wait.until {@detail.getNthFromMeMessage(1) == @new_message_50 + @new_message_50 }
      expect(@detail.getNthFromMeMessage(1)).to eq(@new_message_50 + @new_message_50)
    end

    it "should verify Messages Area, Message, timestamp" do
      expect(@detail.message_headers).to eq("From VA:\nFrom Me:")
      expect(@detail.getNthFromMeMessage(1)).to eq(@new_message_50 + @new_message_50)
      expect(isDateFormatValid?(@detail.getNthFromMeMeTimeStamp(1), "%m/%d/%Y @ %H:%M")).to eq(true)
    end

    it "Users can only see the text box, character countdown and send options, if status is Submitted and veteran has sent 0 or 1 message.  Text description for requests in Cacelled or Not Booked; Historical messages are maintained." do
      cancelRequest
      #textArea box is not available for Cancelled
      expect(@detail.status_element.text).to eq("Status: Cancelled")
      expect(@detail.new_message_element.visible?).to eq(false)
      expect(@detail.message_counter_element.visible?).to eq(false)
      expect(@detail.send_btn_element.visible?).to eq(false)
      expect(@detail.message_desc).to eq("Messages cannot be sent for requests in Cancelled or Not Booked status. If you need additional information, please call your local VA Medical Center.")
      #historical messages maintained
      expect(@detail.messages_all_elements.length).to be >= 1

    end

    it "Message section when request is in not booked status" do
      updateAppointmentRequestedLastUpdatedDate(getDateNthDaysAgo(0, "%Y-%m-%d"), "#{@patientID2}", MENTAL_HEALTH , "Not Booked")
      button_element = @driver.button(:id => 'back-btn')
      @driver.execute_script("$(arguments[0]).click();",button_element)
      waitTillRequestsDisplay
      @driver.refresh
      #@landing_appointments.refresh
      @common.waitWhileSpinnerPresent
      request_details = @landing_appointments.request_table_list_elements[0]
      request_details.scroll.to :top
      @landing_appointments.request_table_list_elements[0].click
      @common.waitWhileSpinnerPresent
      @detail.status_element.when_present(TIME_OUT_LIMIT)
      expect(@detail.has_expected_title?).to eq(true)
      #textArea box is not available for Cancelled
      expect(@detail.status_element.text).to eq("Status: Not Booked")
      expect(@detail.new_message_element.visible?).to eq(false)
      expect(@detail.message_counter_element.visible?).to eq(false)
      expect(@detail.send_btn_element.visible?).to eq(false)
      expect(@detail.message_desc).to eq("Messages cannot be sent for requests in Cancelled or Not Booked status. If you need additional information, please call your local VA Medical Center.")
      #historical messages maintained
      expect(@detail.messages_all_elements.length).to be >= 1
    end

    it "No messages were exchanged on this request - add a new request" do
      button_element = @driver.button(:id => 'back-btn')
      @driver.execute_script("$(arguments[0]).click();",button_element)
      waitTillRequestsDisplay

      #add clerk Mental Health
      typeOfCare = MENTAL_HEALTH
      schedulingMethod = "clerk"
      facilityId = "523"
      facilityName = "BOSTON HCS VAMC"
      typeOfVisit = "Office Visit"
      purposeOfVisit = "Medication Concern"
      otherPurposeOfVisit = ""
      desiredDateTime1 = [getDateNthDaysFromNow(10, "%m/%d/%Y"), "AM"]
      desiredDateTime2 = [getDateNthDaysFromNow(12, "%m/%d/%Y"), "PM"]
      desiredDAteTime3 = [getDateNthDaysFromNow(13, "%m/%d/%Y"), "AM"]
      phoneNum = "(234) 567-8901"
      verifyPhoneNum = "(234) 567-8901"

      bestTimeToCall = " Evening"
      call_morning = false
      call_afternoon = false
      call_evening = true

      bestTimesForVAtoCall = [call_morning, call_afternoon, call_evening]
      vaToCallYou = false
      message = ""
      sendMeEmail = "false"
      email = "test1@hotmail.com"

      addRequest(schedulingMethod, typeOfCare, facilityId, typeOfVisit, purposeOfVisit, otherPurposeOfVisit, desiredDateTime1, desiredDateTime2, desiredDAteTime3, phoneNum, verifyPhoneNum, bestTimesForVAtoCall, vaToCallYou, message, sendMeEmail, email  )
      verifyTableList(facilityName, @today, typeOfCare)
    end

    it "should verify message section for a cancelled request when no message exchange have happened" do

      @landing_appointments.request_table_list_elements[0].click
      @common.waitWhileSpinnerPresent
      @detail.request_details_group_element.when_present(TIME_OUT_LIMIT)
      cancelRequest
      Watir::Wait.until {  @detail.message_desc== "No messages were exchanged on this request." }
      expect(@detail.message_desc).to eq("No messages were exchanged on this request.")
      expect(@detail.messages_all_elements.length).to eq(0)

    end

    it "Should verify message secction when no messages are exchanged for request in not booked status" do

      updateAppointmentRequestedLastUpdatedDate(getDateNthDaysAgo(0, "%Y-%m-%d"), "#{@patientID2}", MENTAL_HEALTH, "Not Booked")
      executeQuery("delete from vardb.appointment_request_message where sender_id='#{@patientID}'")
      button_element = @driver.button(:id => 'back-btn')
      @driver.execute_script("$(arguments[0]).click();",button_element)
      waitTillRequestsDisplay
      #@landing_appointments.refresh
      @driver.refresh
      waitTillRequestsDisplay
      Watir::Wait.until {@landing_appointments.getRequestTextForRowCol(1,2) == "Not Booked"}
      @landing_appointments.request_table_list_elements[1].click
      @common.waitWhileSpinnerPresent
      @detail.status_element.when_present(TIME_OUT_LIMIT)
      expect(@detail.has_expected_title?).to eq(true)
      expect(@detail.status_element.text).to eq("Status: Not Booked")
      Watir::Wait.until {  @detail.message_desc== "No messages were exchanged on this request." }
      expect(@detail.message_desc).to eq("No messages were exchanged on this request.")
      expect(@detail.messages_all_elements.length).to eq(0)
    end

    it "Veteran has sent two messages, textbox, character counter, send button are hidden: add a new request " do
      button_element = @driver.button(:id => 'back-btn')
      @driver.execute_script("$(arguments[0]).click();",button_element)
      waitTillRequestsDisplay

      #add clerk Mental Health
      typeOfCare = MENTAL_HEALTH
      schedulingMethod = "clerk"
      facilityId = "523"
      facilityName = "BOSTON HCS VAMC"
      typeOfVisit = "Office Visit"
      purposeOfVisit = "Medication Concern"
      otherPurposeOfVisit = ""
      desiredDateTime1 = [getDateNthDaysFromNow(20, "%m/%d/%Y"), "AM"]
      desiredDateTime2 = [getDateNthDaysFromNow(22, "%m/%d/%Y"), "PM"]
      desiredDAteTime3 = [getDateNthDaysFromNow(23, "%m/%d/%Y"), "AM"]
      phoneNum = "(234) 567-8901"
      verifyPhoneNum = "(234) 567-8901"

      bestTimeToCall = " Evening"
      call_morning = false
      call_afternoon = false
      call_evening = true

      bestTimesForVAtoCall = [call_morning, call_afternoon, call_evening]
      vaToCallYou = false
      message = ""
      sendMeEmail = "false"
      email = "test1@hotmail.com"

      addRequest(schedulingMethod, typeOfCare, facilityId, typeOfVisit, purposeOfVisit, otherPurposeOfVisit, desiredDateTime1, desiredDateTime2, desiredDAteTime3, phoneNum, verifyPhoneNum, bestTimesForVAtoCall, vaToCallYou, message, sendMeEmail, email  )
      verifyTableList(facilityName, @today, typeOfCare)
    end

    it "Verify sending first message" do
      @landing_appointments.request_table_list_elements[0].click
      @detail.status_element.when_present(TIME_OUT_LIMIT)
      @detail.new_message_element.when_present(TIME_OUT_LIMIT)
      @detail.new_message = "Sending a consecutive message to a clerk"
      @detail.send_btn
      @detail.message_section_element.when_present(TIME_OUT_LIMIT)
      Watir::Wait.until {@detail.getNthFromMeMessage(1) == "Sending a consecutive message to a clerk" }
      expect(@detail.getNthFromMeMessage(1)).to eq("Sending a consecutive message to a clerk")
    end

    it "Verify sending second message" do
      @detail.new_message_element.when_present(TIME_OUT_LIMIT)
      @detail.new_message = "Sending next message"
      @detail.send_btn
      @common.waitWhileSpinnerPresent
      @detail.message_section_element.when_present(TIME_OUT_LIMIT)
      Watir::Wait.until {@detail.getNthFromMeMessage(1) == "Sending next message" }
      expect(@detail.getNthFromMeMessage(1)).to eq("Sending next message")
      expect(@detail.new_message_element.visible?).to eq(false)
      expect(@detail.message_counter_element.visible?).to eq(false)
      expect(@detail.send_btn_element.visible?).to eq(false)
      expect(@detail.message_desc).to eq("You have reached your two message maximum. If you need additional information, please call your local VA Medical Center")
    end

  end

end






































